package gui;

import phonebook.*; // MSOE. Dr. Yoder. 02 February 2016.
import policy.Handler;

import javax.swing.*;
import java.awt.event.*;

/**
 * The main window for the phone book application.
 */
public class BookFrame extends JFrame {
   private Handler bookHandler = new Handler();
   private JList<Entry> list = new JList<>(new DefaultListModel<>());
   private EntryOptionPane entryOptionPane = new EntryOptionPane();

   /**
    * Launch the application
    */
   public static void main(String[] ignored) {
      new BookFrame();
   }

   public BookFrame() {
      setTitle("Phone Book ");
      setDefaultCloseOperation(EXIT_ON_CLOSE);
      setSize(300, 200);

      setupMenus();

      add(list);
      populateList();

      list.addMouseListener(new MouseAdapter() {
         public void mouseClicked(MouseEvent e) {
            if (e.getClickCount() == 2) {
               int index = list.locationToIndex(e.getPoint());
               editItem(index);
            }
         }
      });

      list.addKeyListener(new KeyAdapter() {
         /**
          * Invoked when a key has been typed.
          * This event occurs when a key press is followed by a key release.
          * @param e event holding which key was pressed
          */
         @Override
         public void keyTyped(KeyEvent e) {
            if (KeyEvent.VK_DELETE == e.getKeyChar()) {
               deleteItem();
            } else if (KeyEvent.VK_ENTER == e.getKeyChar()) {
               editItem();
            } else if ((e.getKeyChar() == 'N') && ((e.getModifiers() & KeyEvent.CTRL_MASK) != 0)) {
               addItem();
            }
         }
      });

      setVisible(true);
   }

   private void populateList() {
      bookHandler.addEntry("Babbage, Charles", "123-4567");
      bookHandler.addEntry("Liskov, Barbara", "123-7654");
      bookHandler.addEntry("Turing, Alan", "123-6382");
      bookHandler.addEntry("Kilby, Jack", "123-9876");
      bookHandler.addEntry("Hopper, Grace", "123-8765");
      refreshList();
   }

   private void setupMenus() {
      JMenuBar bar = new JMenuBar();
      setJMenuBar(bar);
      JMenu menu = new JMenu("Edit");
      bar.add(menu);

      JMenuItem item = new JMenuItem("Add Item (Ctrl-A)");
      menu.add(item);
      item.addActionListener(e -> addItem());

      item = new JMenuItem("Edit Item (ENTER)");
      menu.add(item);
      item.addActionListener(e -> editItem());

      item = new JMenuItem("Delete Item (DEL)");
      menu.add(item);
      item.addActionListener(e -> deleteItem());

      item = new JMenuItem("Undo");
      menu.add(item);
      item.addActionListener(e -> undoOperation());

      item = new JMenuItem("Redo");
      menu.add(item);
      item.addActionListener(e -> redoOperation());
   }

   /**
    * Prompt user for a new entry, and add it to the phone book.
    */
   private void addItem() {
      //How to show multiple fields in JOptionPanee
      //http://stackoverflow.com/questions/6555040/multiple-input-in-joptionpane-showinputdialog
      int result = entryOptionPane.showModalDialog();
      if (result == JOptionPane.OK_OPTION) {
         bookHandler.addEntry(entryOptionPane.getName(), entryOptionPane.getPhone());
         refreshList();
      }
   }

   /**
    * Edit the currently-selected item.
    */
   private void editItem() {
      int index = list.getSelectedIndex();
      if (index >= 0) {
         editItem(index);
      }
   }

   /**
    * Edit item at a given location in the list
    *
    * @param index index of the item to edit. Must be a valid index.
    */
   private void editItem(int index) {
      DefaultListModel<Entry> defaultModel = getDefaultListModel();
      Entry entry = defaultModel.getElementAt(index);
      int result = entryOptionPane.showModalDialog(entry);
      if (result == JOptionPane.OK_OPTION) {
         bookHandler.removeEntry(entry.name());
         bookHandler.addEntry(entryOptionPane.getName(), entryOptionPane.getPhone());
         refreshList();
      }
   }

   /**
    * Delete the currently-selected item.
    */
   private void deleteItem() {
      int index = list.getSelectedIndex();
      if (index >= 0) {
         deleteItem(index);
      }
   }

   /**
    * Delete the in the list.
    *
    * @param index index of item to delete within list. Must be a valid index.
    */
   private void deleteItem(int index) {
      DefaultListModel<Entry> defaultModel = getDefaultListModel();
      Entry entry = defaultModel.getElementAt(index);
      bookHandler.removeEntry(entry.name());
      refreshList();
   }

   /**
    * Undoes the effect of the last operation.
    */
   private void undoOperation() {
      bookHandler.undo();
      refreshList();
   }

   /**
    * Redoes the last undone operation.
    */
   private void redoOperation() {
      bookHandler.redo();
      refreshList();
   }

   /**
    * Refresh the JList to match the contents of the model.
    */
   private void refreshList() {
      DefaultListModel<Entry> defaultModel = getDefaultListModel();
      defaultModel.clear();
      for (Entry entry : bookHandler.getPhoneBook()) {
         System.out.println("DEBUG: Displaying entry " + entry);
         defaultModel.insertElementAt(entry, defaultModel.size());
      }
      repaint();
      System.out.println("Debugging.");
   }

   /**
    * Get the internal list model
    *
    * @return the DefaultListModel for the JList list.
    * @throws RuntimeException if the model is not a DefaultListModel. This should never
    *                          happen, because we specify to use a DefaultListModel when constructing the JList list.
    */
   private DefaultListModel<Entry> getDefaultListModel() throws RuntimeException {
      ListModel<Entry> model = list.getModel();
      if (!(model instanceof DefaultListModel)) {
         throw new RuntimeException("This should never happen because we give a DefaultListModel to the list when we create it.");
      }
      return (DefaultListModel<Entry>) model;
   }
}
